// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_SSL_WRAPPER_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_SSL_WRAPPER_H

#include <openssl/asn1.h>

#include "common.h"

enum AuthenticationState {
    AUTHENTICATION_UNINITIALIZED,
    AUTHENTICATION_NOT_STARTED,
    AUTHENTICATION_IN_PROGRESS,
    AUTHENTICATION_SUCCESS,
    AUTHENTICATION_FAILURE
};

struct SslWrapperState;

/**
 * A wrapper around OpenSSL. In the extremely unlikely case that your platform does not
 * have an available OpenSSL port, you will need to customize this class. Has the
 * necessary callbacks set up to make it thread safe.
 */
class SslWrapper {
public:
    SslWrapper();
    ~SslWrapper();

    void shutdown();
    /**
     * @internal
     * Initialize this ssl connection object. We imagine that there should be only
     * one of these objects and init should be called prior to starting the usb
     * connection.
     * @param rootCert A root certificate in .pem format. This is used to validate the
     *        server's certificate.
     * @param clientCert A client certificate that will be used in the ssl handshake.
     *        For simplicity, we enforce that this certificate should trace back to the
     *        same root certificate as that of the server.
     * @param clientKey The private key that corresponds to the certificate clientCert.
     * @return true If everything was initialized properly, false otherwise. You should
     *         not attempt to continue if this method returns false.
     */
    bool init(const std::string& rootCert, const std::string& clientCert,
            const std::string& clientKey);
    /**
     * @internal
     * Take the next step in the ssl handshake.
     * @param in A message that was received from the other side. NULL if initiating
     *           a new shandshake as a client.
     * @param inLen The length of the message passed in.
     * @param out Populated if this handshake produced more data. It already has the
     *        correct GAL header put in so there is no need to tack on anything in front
     *        of it.
     * @return 0 If the connection was shutdown due to an authentication failure.
     *         1 If successfully authenticated.
     *       < 0 If more steps are needed to arrive at a conclusion.
     */
    int handshake(const void* in, size_t inLen, IoBuffer* out);
    /**
     * @internal
     * Check if the peer on this ssl connection is trusted. This should generally be called
     * after handshake has succeeded to check if the other end says it is who it is.
     * @return STATUS_SUCCESS on success or a STATUS_AUTHENTICATION_FAILURE code on failure.
     */
    int verifyPeer();
    /**
     * @internal
     * Queue plaintext for encryption. Call |encryptionPipelineDequeue| to
     * retrieve the resulting ciphertext.
     * @param data The plaintext
     * @param len The length of the plaintext.
     * @return > 0 on success with the resulting number of ciphertext bytes.
     *         <= 0 on error.
     */
    int encryptionPipelineEnqueue(void* data, size_t len);
    /**
     * @internal
     * Get the ciphertext for the data that was previously enqueued by
     * |encryptionPipelineEnqueue|.
     * @param data The output buffer.
     * @param len The size of the output buffer.
     * @return 0 If there is no more ciphertext to read at this time.
     *         < 0 on error
     *         otherwise the number of bytes read.
     */
    int encryptionPipelineDequeue(void* data, size_t len);
    /**
     * @internal
     * Enqueue a block of data for decryption. Use the corresponding dequeue function
     * to get the plaintext.
     * @param data Pointer to the data.
     * @param len the length of the buffer.
     * @return true on success and false on error. You should call
     *         decryptionPipelineDequeue until all the bytes have been read.
     */
    bool decryptionPipelineEnqueue(void* data, size_t len);
    /**
     * @internal
     * Dequeue a block of plaintext that was queued for decryption by
     * |decryptionPipelineEnqueue|.
     * @param data The output buffer.
     * @param len The size of the output buffer.
     * @return 0 If there is no more plaintext to read at this time.
     *         < 0 on error
     *         otherwise the number of bytes read.
     */
    int decryptionPipelineDequeue(void* data, size_t len);
    /**
     * @internal
     * Used to make OpenSSL use the supplied time instead of the value from gettimeofday().
     * @param time Milliseconds since epoch.
     */
    void setCertificateVerificationTime(time_t time) {
        mOverrideTime = time;
        mHasOverrideTime = true;
    }

#if FIX_ASN1_UTCTIME
    /**
     * @internal
     * Corrects the ASN1 UTC TIME in certificate as required by
     * boringssl. Boringssl expects the length of the ASN1 UTC TIME string
     * to be fixed length, instead of earlier versions that were flexible.
     */
    static void correctAsn1UtcTime(ASN1_UTCTIME* asn1UtcTime);
#endif

private:
    AuthenticationState mAuthState;
    SslWrapperState *const mState;
    bool mHasOverrideTime;
    time_t mOverrideTime;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_SSL_WRAPPER_H
